library ieee;
use ieee.std_logic_1164.all;
use ieee.numeric_std.all;
use ieee.math_real.all;

entity fsmd is
  port( clk  : in  std_logic;
        rst  : in  std_logic;
        go   : in  std_logic;
        done : out std_logic);
end fsmd;

architecture bhv of fsmd is

  type STATE_TYPE is (S_START, S_COUNT, S_DONE);
  signal state, next_state : STATE_TYPE;

  -- a nice way of determining exactly how many bits are needed by count
  
  constant MAX_COUNT_VAL     : natural := 10;
  constant MAX_COUNT_BITS    : natural := integer(ceil(log2(real(MAX_COUNT_VAL))));
  signal   count, next_count : unsigned(MAX_COUNT_BITS-1 downto 0);
  
begin

  process (clk, rst)
  begin
    if (rst = '1') then
      state <= S_START;
    elsif (clk = '1' and clk'event) then
      state <= next_state;
      count <= next_count;
    end if;
  end process;

  -- solution:
  -- make process sensitive to count, but only assign values to next_count
  -- notice that this is the exact same way we are handling the state

  process(go, state, count)
  begin

    case state is
      when S_START =>

        done       <= '0';
        next_count <= to_unsigned(1, count'length);

        if (go = '0') then
          next_state <= S_START;
        else
          next_state <= S_COUNT;
        end if;

      when S_COUNT =>

        done       <= '0';
        next_count <= count + 1;

        if (count = MAX_COUNT_VAL) then
          next_state <= S_DONE;
        else
          next_state <= S_COUNT;
        end if;

      when S_DONE =>

        next_count <= to_unsigned(MAX_COUNT_VAL, count'length);
        done       <= '1';
        next_state <= S_DONE;

      when others => null;
    end case;

  end process;
end bhv;
